//===-- Half-precision acosf16(x) function --------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception.
//
//
//===----------------------------------------------------------------------===//

#include "src/math/acosf16.h"
#include "hdr/errno_macros.h"
#include "hdr/fenv_macros.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/sqrt.h"
#include "src/__support/macros/optimization.h"

namespace LIBC_NAMESPACE_DECL {

// Generated by Sollya using the following command:
// > round(pi/2, SG, RN);
// > round(pi, SG, RN);
static constexpr float PI_OVER_2 = 0x1.921fb6p0f;
static constexpr float PI = 0x1.921fb6p1f;

#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
static constexpr size_t N_EXCEPTS = 2;

static constexpr fputil::ExceptValues<float16, N_EXCEPTS> ACOSF16_EXCEPTS{{
    // (input, RZ output, RU offset, RD offset, RN offset)
    {0xacaf, 0x3e93, 1, 0, 0},
    {0xb874, 0x4052, 1, 0, 1},
}};
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

LLVM_LIBC_FUNCTION(float16, acosf16, (float16 x)) {
  using FPBits = fputil::FPBits<float16>;
  FPBits xbits(x);

  uint16_t x_u = xbits.uintval();
  uint16_t x_abs = x_u & 0x7fff;
  uint16_t x_sign = x_u >> 15;

  // |x| > 0x1p0, |x| > 1, or x is NaN.
  if (LIBC_UNLIKELY(x_abs > 0x3c00)) {
    // acosf16(NaN) = NaN
    if (xbits.is_nan()) {
      if (xbits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }

      return x;
    }

    // 1 < |x| <= +/-inf
    fputil::raise_except_if_required(FE_INVALID);
    fputil::set_errno_if_required(EDOM);

    return FPBits::quiet_nan().get_val();
  }

  float xf = x;

#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
  // Handle exceptional values
  if (auto r = ACOSF16_EXCEPTS.lookup(x_u); LIBC_UNLIKELY(r.has_value()))
    return r.value();
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

  // |x| == 0x1p0, x is 1 or -1
  // if x is (-)1, return pi, else
  // if x is (+)1, return 0
  if (LIBC_UNLIKELY(x_abs == 0x3c00))
    return fputil::cast<float16>(x_sign ? PI : 0.0f);

  float xsq = xf * xf;

  // |x| <= 0x1p-1, |x| <= 0.5
  if (x_abs <= 0x3800) {
    // if x is 0, return pi/2
    if (LIBC_UNLIKELY(x_abs == 0))
      return fputil::cast<float16>(PI_OVER_2);

    // Note that: acos(x) = pi/2 + asin(-x) = pi/2 - asin(x)
    // Degree-6 minimax polynomial of asin(x) generated by Sollya with:
    // > P = fpminimax(asin(x)/x, [|0, 2, 4, 6, 8|], [|SG...|], [0, 0.5]);
    float interm =
        fputil::polyeval(xsq, 0x1.000002p0f, 0x1.554c2ap-3f, 0x1.3541ccp-4f,
                         0x1.43b2d6p-5f, 0x1.a0d73ep-5f);
    return fputil::cast<float16>(fputil::multiply_add(-xf, interm, PI_OVER_2));
  }

  // When |x| > 0.5, assume that 0.5 < |x| <= 1
  //
  // Step-by-step range-reduction proof:
  // 1:  Let y = asin(x), such that, x = sin(y)
  // 2:  From complimentary angle identity:
  //       x = sin(y) = cos(pi/2 - y)
  // 3:  Let z = pi/2 - y, such that x = cos(z)
  // 4:  From double angle formula; cos(2A) = 1 - 2 * sin^2(A):
  //       z = 2A, z/2 = A
  //       cos(z) = 1 - 2 * sin^2(z/2)
  // 5:  Make sin(z/2) subject of the formula:
  //       sin(z/2) = sqrt((1 - cos(z))/2)
  // 6:  Recall [3]; x = cos(z). Therefore:
  //       sin(z/2) = sqrt((1 - x)/2)
  // 7:  Let u = (1 - x)/2
  // 8:  Therefore:
  //       asin(sqrt(u)) = z/2
  //       2 * asin(sqrt(u)) = z
  // 9:  Recall [3]; z = pi/2 - y. Therefore:
  //       y = pi/2 - z
  //       y = pi/2 - 2 * asin(sqrt(u))
  // 10: Recall [1], y = asin(x). Therefore:
  //       asin(x) = pi/2 - 2 * asin(sqrt(u))
  // 11: Recall that: acos(x) = pi/2 + asin(-x) = pi/2 - asin(x)
  //     Therefore:
  //       acos(x) = pi/2 - (pi/2 - 2 * asin(sqrt(u)))
  //       acos(x) = 2 * asin(sqrt(u))
  //
  // THE RANGE REDUCTION, HOW?
  // 12: Recall [7], u = (1 - x)/2
  // 13: Since 0.5 < x <= 1, therefore:
  //       0 <= u <= 0.25 and 0 <= sqrt(u) <= 0.5
  //
  // Hence, we can reuse the same [0, 0.5] domain polynomial approximation for
  // Step [11] as `sqrt(u)` is in range.
  // When -1 < x <= -0.5, the identity:
  //       acos(x) = pi - acos(-x)
  // allows us to compute for the negative x value (lhs)
  // with a positive x value instead (rhs).

  float xf_abs = (xf < 0 ? -xf : xf);
  float u = fputil::multiply_add(-0.5f, xf_abs, 0.5f);
  float sqrt_u = fputil::sqrt<float>(u);

  // Degree-6 minimax polynomial of asin(x) generated by Sollya with:
  // > P = fpminimax(asin(x)/x, [|0, 2, 4, 6, 8|], [|SG...|], [0, 0.5]);
  float asin_sqrt_u =
      sqrt_u * fputil::polyeval(u, 0x1.000002p0f, 0x1.554c2ap-3f,
                                0x1.3541ccp-4f, 0x1.43b2d6p-5f, 0x1.a0d73ep-5f);

  return fputil::cast<float16>(
      x_sign ? fputil::multiply_add(-2.0f, asin_sqrt_u, PI) : 2 * asin_sqrt_u);
}
} // namespace LIBC_NAMESPACE_DECL
