/*
 * Copyright (C) 2008, 2013 Apple Inc. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#include "config.h"
#include "CallFrame.h"

#include "CallFrameInlines.h"
#include "CodeBlock.h"
#include "Interpreter.h"
#include "Operations.h"

namespace JSC {

#ifndef NDEBUG
JSStack* CallFrame::stack()
{
    return &interpreter()->stack();
}

#endif

#if USE(JSVALUE32_64)
unsigned CallFrame::locationAsBytecodeOffset() const
{
    ASSERT(codeBlock());
    ASSERT(hasLocationAsBytecodeOffset());
    return currentVPC() - codeBlock()->instructions().begin();
}

void CallFrame::setLocationAsBytecodeOffset(unsigned offset)
{
    ASSERT(codeBlock());
    setCurrentVPC(codeBlock()->instructions().begin() + offset);
    ASSERT(hasLocationAsBytecodeOffset());
}
#else
Instruction* CallFrame::currentVPC() const
{
    return codeBlock()->instructions().begin() + locationAsBytecodeOffset();
}
void CallFrame::setCurrentVPC(Instruction* vpc)
{
    setLocationAsBytecodeOffset(vpc - codeBlock()->instructions().begin());
}
#endif
    
#if ENABLE(DFG_JIT)
unsigned CallFrame::bytecodeOffsetFromCodeOriginIndex()
{
    ASSERT(hasLocationAsCodeOriginIndex());
    CodeBlock* codeBlock = this->codeBlock();
    ASSERT(codeBlock);

    CodeOrigin codeOrigin;
    unsigned index = locationAsCodeOriginIndex();
    ASSERT(codeBlock->canGetCodeOrigin(index));
    codeOrigin = codeBlock->codeOrigin(index);

    for (InlineCallFrame* inlineCallFrame = codeOrigin.inlineCallFrame; inlineCallFrame;) {
        if (inlineCallFrame->baselineCodeBlock() == codeBlock)
            return codeOrigin.bytecodeIndex;

        codeOrigin = inlineCallFrame->caller;
        inlineCallFrame = codeOrigin.inlineCallFrame;
    }
    return codeOrigin.bytecodeIndex;
}

#endif // ENABLE(DFG_JIT)

Register* CallFrame::frameExtentInternal()
{
    CodeBlock* codeBlock = this->codeBlock();
    ASSERT(codeBlock);
    return registers() + codeBlock->m_numCalleeRegisters;
}

StackIterator CallFrame::begin(StackIterator::FrameFilter filter)
{
    ASSERT(this);
    return StackIterator(this, filter);
}

StackIterator CallFrame::find(JSFunction* calleeFunctionObj, StackIterator::FrameFilter filter)
{
    ASSERT(this);
    StackIterator iter = StackIterator(this, filter);
    iter.find(calleeFunctionObj);
    return iter;
}

StackIterator::Frame* CallFrame::end()
{
    return StackIterator::end();
}

}
