/*	$OpenBSD: hash_test.c,v 1.3 2025/08/02 06:05:13 tb Exp $ */

/*
 * Copyright (c) 2025 Theo Buehler <tb@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>

#include <md5.h>
#include <rmd160.h>
#include <sha1.h>
#include <sha2.h>
#include <stdio.h>
#include <string.h>

#define MAX_DIGEST_LENGTH SHA512_DIGEST_LENGTH

struct hash_test_case {
	const char	*in;
	const uint8_t	 out[MAX_DIGEST_LENGTH];
};

enum {
	hash_md5,
	hash_rmd160,
	hash_sha1,
	hash_sha224,
	hash_sha256,
	hash_sha384,
	hash_sha512,
	hash_sha512_256,
	NUM_HASHES,
};

/* RFC 1321, Appendix A.5 */
static const struct hash_test_case md5_tests[] = {
	{
		.out = {
			0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
			0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e,
		},
	},
	{
		.in = "",
		.out = {
			0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
			0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e,
		},
	},
	{
		.in = "a",
		.out = {
			0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8,
			0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61,
		},
	},
	{
		.in = "abc",
		.out = {
			0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0,
			0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72,
		},
	},
	{
		.in = "message digest",
		.out = {
			0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d,
			0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0,
		},
	},
	{
		.in = "abcdefghijklmnopqrstuvwxyz",
		.out = {
			0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00,
			0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b,
		},
	},
	{
		.in = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		    "abcdefghijklmnopqrstuvwxyz0123456789",
		.out = {
			0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5,
			0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f,
		},
	},
	{
		.in = "1234567890123456789012345678901234567890"
		    "1234567890123456789012345678901234567890",
		.out = {
			0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55,
			0xac, 0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a,
		},
	},
};

#define N_MD5_TESTS (sizeof(md5_tests) / sizeof(md5_tests[0]))

static void
md5_init(void *ctx)
{
	MD5Init(ctx);
}

static void
md5_update(void *ctx, const uint8_t *data, size_t len)
{
	MD5Update(ctx, data, len);
}

static void
md5_final(void *digest, void *ctx)
{
	MD5Final(digest, ctx);
}

/* https://homes.esat.kuleuven.be/~bosselae/ripemd160.html */
static const struct hash_test_case rmd160_tests[] = {
	{
		.out = {
			0x9c, 0x11, 0x85, 0xa5, 0xc5, 0xe9, 0xfc, 0x54,
			0x61, 0x28, 0x08, 0x97, 0x7e, 0xe8, 0xf5, 0x48,
			0xb2, 0x25, 0x8d, 0x31,
		},
	},
	{
		.in = "",
		.out = {
			0x9c, 0x11, 0x85, 0xa5, 0xc5, 0xe9, 0xfc, 0x54,
			0x61, 0x28, 0x08, 0x97, 0x7e, 0xe8, 0xf5, 0x48,
			0xb2, 0x25, 0x8d, 0x31,
		},
	},
	{
		.in = "a",
		.out = {
			0x0b, 0xdc, 0x9d, 0x2d, 0x25, 0x6b, 0x3e, 0xe9,
			0xda, 0xae, 0x34, 0x7b, 0xe6, 0xf4, 0xdc, 0x83,
			0x5a, 0x46, 0x7f, 0xfe,
		},
	},
	{
		.in = "abc",
		.out = {
			0x8e, 0xb2, 0x08, 0xf7, 0xe0, 0x5d, 0x98, 0x7a,
			0x9b, 0x04, 0x4a, 0x8e, 0x98, 0xc6, 0xb0, 0x87,
			0xf1, 0x5a, 0x0b, 0xfc,
		},
	},
	{
		.in = "message digest",
		.out = {
			0x5d, 0x06, 0x89, 0xef, 0x49, 0xd2, 0xfa, 0xe5,
			0x72, 0xb8, 0x81, 0xb1, 0x23, 0xa8, 0x5f, 0xfa,
			0x21, 0x59, 0x5f, 0x36,
		},
	},
	{
		.in = "abcdefghijklmnopqrstuvwxyz",
		.out = {
			0xf7, 0x1c, 0x27, 0x10, 0x9c, 0x69, 0x2c, 0x1b,
			0x56, 0xbb, 0xdc, 0xeb, 0x5b, 0x9d, 0x28, 0x65,
			0xb3, 0x70, 0x8d, 0xbc,
		},
	},
	{
		.in = "abcdbcdecdefdefgefghfghighijhijkijkljkl"
		    "mklmnlmnomnopnopq",
		.out = {
			0x12, 0xa0, 0x53, 0x38, 0x4a, 0x9c, 0x0c, 0x88,
			0xe4, 0x05, 0xa0, 0x6c, 0x27, 0xdc, 0xf4, 0x9a,
			0xda, 0x62, 0xeb, 0x2b,
		},
	},
	{
		.in = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
		    "0123456789",
		.out = {
			0xb0, 0xe2, 0x0b, 0x6e, 0x31, 0x16, 0x64, 0x02,
			0x86, 0xed, 0x3a, 0x87, 0xa5, 0x71, 0x30, 0x79,
			0xb2, 0x1f, 0x51, 0x89,
		},
	},
	{
		.in = "1234567890123456789012345678901234567890"
		    "1234567890123456789012345678901234567890",
		.out = {
			0x9b, 0x75, 0x2e, 0x45, 0x57, 0x3d, 0x4b, 0x39,
			0xf4, 0xdb, 0xd3, 0x32, 0x3c, 0xab, 0x82, 0xbf,
			0x63, 0x32, 0x6b, 0xfb,
		},
	},
};

#define N_RMD160_TESTS (sizeof(rmd160_tests) / sizeof(rmd160_tests[0]))

static void
rmd160_init(void *ctx)
{
	RMD160Init(ctx);
}

static void
rmd160_update(void *ctx, const uint8_t *data, size_t len)
{
	RMD160Update(ctx, data, len);
}

static void
rmd160_final(void *digest, void *ctx)
{
	RMD160Final(digest, ctx);
}

/* RFC 3174 - Appendix A (plus two zero-length tests) */
static const struct hash_test_case sha1_tests[] = {
	{
		.out = {
			0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d,
			0x32, 0x55, 0xbf, 0xef, 0x95, 0x60, 0x18, 0x90,
			0xaf, 0xd8, 0x07, 0x09,
		},
	},
	{
		.in = "",
		.out = {
			0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d,
			0x32, 0x55, 0xbf, 0xef, 0x95, 0x60, 0x18, 0x90,
			0xaf, 0xd8, 0x07, 0x09,
		},
	},
	{
		.in = "abc",
		.out = {
			0xa9, 0x99, 0x3e, 0x36, 0x47, 0x06, 0x81, 0x6a,
			0xba, 0x3e, 0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c,
			0x9c, 0xd0, 0xd8, 0x9d,
		},
	},
	{
		.in = "abcdbcdecdefdefgefghfghighijhi"
		    "jkijkljklmklmnlmnomnopnopq",
		.out = {
			0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e,
			0xba, 0xae, 0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5,
			0xe5, 0x46, 0x70, 0xf1,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0xde, 0xa3, 0x56, 0xa2, 0xcd, 0xdd, 0x90, 0xc7,
			0xa7, 0xec, 0xed, 0xc5, 0xeb, 0xb5, 0x63, 0x93,
			0x4f, 0x46, 0x04, 0x52,
		},
	},
};

#define N_SHA1_TESTS (sizeof(sha1_tests) / sizeof(sha1_tests[0]))

static void
sha1_init(void *ctx)
{
	SHA1Init(ctx);
}

static void
sha1_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA1Update(ctx, data, len);
}

static void
sha1_final(void *digest, void *ctx)
{
	SHA1Final(digest, ctx);
}

static const struct hash_test_case sha224_tests[] = {
	{
		.out = {
			0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9,
			0x47, 0x61, 0x02, 0xbb, 0x28, 0x82, 0x34, 0xc4,
			0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a,
			0xc5, 0xb3, 0xe4, 0x2f,
		},
	},
	{
		.in = "",
		.out = {
			0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9,
			0x47, 0x61, 0x02, 0xbb, 0x28, 0x82, 0x34, 0xc4,
			0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a,
			0xc5, 0xb3, 0xe4, 0x2f,
		},
	},
	{
		.in = "abc",
		.out = {
			0x23, 0x09, 0x7d, 0x22, 0x34, 0x05, 0xd8, 0x22,
			0x86, 0x42, 0xa4, 0x77, 0xbd, 0xa2, 0x55, 0xb3,
			0x2a, 0xad, 0xbc, 0xe4, 0xbd, 0xa0, 0xb3, 0xf7,
			0xe3, 0x6c, 0x9d, 0xa7,
		},
	},
	{
		.in = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.out = {
			0x75, 0x38, 0x8b, 0x16, 0x51, 0x27, 0x76, 0xcc,
			0x5d, 0xba, 0x5d, 0xa1, 0xfd, 0x89, 0x01, 0x50,
			0xb0, 0xc6, 0x45, 0x5c, 0xb4, 0xf5, 0x8b, 0x19,
			0x52, 0x52, 0x25, 0x25,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0x56, 0x7f, 0x69, 0xf1, 0x68, 0xcd, 0x78, 0x44,
			0xe6, 0x52, 0x59, 0xce, 0x65, 0x8f, 0xe7, 0xaa,
			0xdf, 0xa2, 0x52, 0x16, 0xe6, 0x8e, 0xca, 0x0e,
			0xb7, 0xab, 0x82, 0x62,
		},
	},
	{
		.in = "\x07",
		.out = {
			0x00, 0xec, 0xd5, 0xf1, 0x38, 0x42, 0x2b, 0x8a,
			0xd7, 0x4c, 0x97, 0x99, 0xfd, 0x82, 0x6c, 0x53,
			0x1b, 0xad, 0x2f, 0xca, 0xbc, 0x74, 0x50, 0xbe,
			0xe2, 0xaa, 0x8c, 0x2a,
		},
	},
};

#define N_SHA224_TESTS (sizeof(sha224_tests) / sizeof(sha224_tests[0]))

static void
sha224_init(void *ctx)
{
	SHA224Init(ctx);
}

static void
sha224_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA224Update(ctx, data, len);
}

static void
sha224_final(void *digest, void *ctx)
{
	SHA224Final(digest, ctx);
}

static const struct hash_test_case sha256_tests[] = {
	{
		.out = {
			0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
			0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
			0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
			0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55,
		},
	},
	{
		.in = "",
		.out = {
			0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
			0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
			0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
			0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55,
		},
	},
	{
		.in = "abc",
		.out = {
			0xBA, 0x78, 0x16, 0xBF, 0x8F, 0x01, 0xCF, 0xEA,
			0x41, 0x41, 0x40, 0xDE, 0x5D, 0xAE, 0x22, 0x23,
			0xB0, 0x03, 0x61, 0xA3, 0x96, 0x17, 0x7A, 0x9C,
			0xB4, 0x10, 0xFF, 0x61, 0xF2, 0x00, 0x15, 0xAD,
		},
	},
	{
		.in = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.out = {
			0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0x59, 0x48, 0x47, 0x32, 0x84, 0x51, 0xBD, 0xFA,
			0x85, 0x05, 0x62, 0x25, 0x46, 0x2C, 0xC1, 0xD8,
			0x67, 0xD8, 0x77, 0xFB, 0x38, 0x8D, 0xF0, 0xCE,
			0x35, 0xF2, 0x5A, 0xB5, 0x56, 0x2B, 0xFB, 0xB5,
		},
	},
	{
		.in = "\x19",
		.out = {
			0x68, 0xaa, 0x2e, 0x2e, 0xe5, 0xdf, 0xf9, 0x6e,
			0x33, 0x55, 0xe6, 0xc7, 0xee, 0x37, 0x3e, 0x3d,
			0x6a, 0x4e, 0x17, 0xf7, 0x5f, 0x95, 0x18, 0xd8,
			0x43, 0x70, 0x9c, 0x0c, 0x9b, 0xc3, 0xe3, 0xd4,
		},
	},
};

#define N_SHA256_TESTS (sizeof(sha256_tests) / sizeof(sha256_tests[0]))

static void
sha256_init(void *ctx)
{
	SHA256Init(ctx);
}

static void
sha256_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA256Update(ctx, data, len);
}

static void
sha256_final(void *digest, void *ctx)
{
	SHA256Final(digest, ctx);
}

static const struct hash_test_case sha384_tests[] = {
	{
		.out = {
			0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
			0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
			0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
			0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
			0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
			0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b,
		},
	},
	{
		.in = "",
		.out = {
			0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
			0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
			0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
			0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
			0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
			0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b,
		},
	},
	{
		.in = "abc",
		.out = {
			0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
			0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
			0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
			0x1A, 0x8B, 0x60, 0x5A, 0x43, 0xFF, 0x5B, 0xED,
			0x80, 0x86, 0x07, 0x2B, 0xA1, 0xE7, 0xCC, 0x23,
			0x58, 0xBA, 0xEC, 0xA1, 0x34, 0xC8, 0x25, 0xA7,
		},
	},
	{
		.in = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
		    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.out = {
			0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
			0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
			0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
			0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
			0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
			0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0x2f, 0xc6, 0x4a, 0x4f, 0x50, 0x0d, 0xdb, 0x68,
			0x28, 0xf6, 0xa3, 0x43, 0x0b, 0x8d, 0xd7, 0x2a,
			0x36, 0x8e, 0xb7, 0xf3, 0xa8, 0x32, 0x2a, 0x70,
			0xbc, 0x84, 0x27, 0x5b, 0x9c, 0x0b, 0x3a, 0xb0,
			0x0d, 0x27, 0xa5, 0xcc, 0x3c, 0x2d, 0x22, 0x4a,
			0xa6, 0xb6, 0x1a, 0x0d, 0x79, 0xfb, 0x45, 0x96,
		},
	},
	{
		.in = "\xb9",
		.out = {
			0xbc, 0x80, 0x89, 0xa1, 0x90, 0x07, 0xc0, 0xb1,
			0x41, 0x95, 0xf4, 0xec, 0xc7, 0x40, 0x94, 0xfe,
			0xc6, 0x4f, 0x01, 0xf9, 0x09, 0x29, 0x28, 0x2c,
			0x2f, 0xb3, 0x92, 0x88, 0x15, 0x78, 0x20, 0x8a,
			0xd4, 0x66, 0x82, 0x8b, 0x1c, 0x6c, 0x28, 0x3d,
			0x27, 0x22, 0xcf, 0x0a, 0xd1, 0xab, 0x69, 0x38,
		},
	},
};

#define N_SHA384_TESTS (sizeof(sha384_tests) / sizeof(sha384_tests[0]))

static void
sha384_init(void *ctx)
{
	SHA384Init(ctx);
}

static void
sha384_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA384Update(ctx, data, len);
}

static void
sha384_final(void *digest, void *ctx)
{
	SHA384Final(digest, ctx);
}

static const struct hash_test_case sha512_tests[] = {
	{
		.out = {
			0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
			0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
			0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
			0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
			0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
			0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
			0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
			0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e,
		},
	},
	{
		.in = "",
		.out = {
			0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
			0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
			0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
			0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
			0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
			0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
			0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
			0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e,
		},
	},
	{
		.in = "abc",
		.out = {
			0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
			0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
			0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
			0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
			0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
			0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
			0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
			0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f,
		},
	},
	{
		.in = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
		    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.out = {
			0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
			0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
			0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
			0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
			0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
			0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
			0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
			0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0x89, 0xd0, 0x5b, 0xa6, 0x32, 0xc6, 0x99, 0xc3,
			0x12, 0x31, 0xde, 0xd4, 0xff, 0xc1, 0x27, 0xd5,
			0xa8, 0x94, 0xda, 0xd4, 0x12, 0xc0, 0xe0, 0x24,
			0xdb, 0x87, 0x2d, 0x1a, 0xbd, 0x2b, 0xa8, 0x14,
			0x1a, 0x0f, 0x85, 0x07, 0x2a, 0x9b, 0xe1, 0xe2,
			0xaa, 0x04, 0xcf, 0x33, 0xc7, 0x65, 0xcb, 0x51,
			0x08, 0x13, 0xa3, 0x9c, 0xd5, 0xa8, 0x4c, 0x4a,
			0xca, 0xa6, 0x4d, 0x3f, 0x3f, 0xb7, 0xba, 0xe9,
		},
	},
	{
		.in = "\xd0",
		.out = {
			0x99, 0x92, 0x20, 0x29, 0x38, 0xe8, 0x82, 0xe7,
			0x3e, 0x20, 0xf6, 0xb6, 0x9e, 0x68, 0xa0, 0xa7,
			0x14, 0x90, 0x90, 0x42, 0x3d, 0x93, 0xc8, 0x1b,
			0xab, 0x3f, 0x21, 0x67, 0x8d, 0x4a, 0xce, 0xee,
			0xe5, 0x0e, 0x4e, 0x8c, 0xaf, 0xad, 0xa4, 0xc8,
			0x5a, 0x54, 0xea, 0x83, 0x06, 0x82, 0x6c, 0x4a,
			0xd6, 0xe7, 0x4c, 0xec, 0xe9, 0x63, 0x1b, 0xfa,
			0x8a, 0x54, 0x9b, 0x4a, 0xb3, 0xfb, 0xba, 0x15,
		},
	},
};

#define N_SHA512_TESTS (sizeof(sha512_tests) / sizeof(sha512_tests[0]))

static void
sha512_init(void *ctx)
{
	SHA512Init(ctx);
}

static void
sha512_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA512Update(ctx, data, len);
}

static void
sha512_final(void *digest, void *ctx)
{
	SHA512Final(digest, ctx);
}

static const struct hash_test_case sha512_256_tests[] = {
	{
		.out = {
			0xc6, 0x72, 0xb8, 0xd1, 0xef, 0x56, 0xed, 0x28,
			0xab, 0x87, 0xc3, 0x62, 0x2c, 0x51, 0x14, 0x06,
			0x9b, 0xdd, 0x3a, 0xd7, 0xb8, 0xf9, 0x73, 0x74,
			0x98, 0xd0, 0xc0, 0x1e, 0xce, 0xf0, 0x96, 0x7a,
		},
	},
	{
		.in = "",
		.out = {
			0xc6, 0x72, 0xb8, 0xd1, 0xef, 0x56, 0xed, 0x28,
			0xab, 0x87, 0xc3, 0x62, 0x2c, 0x51, 0x14, 0x06,
			0x9b, 0xdd, 0x3a, 0xd7, 0xb8, 0xf9, 0x73, 0x74,
			0x98, 0xd0, 0xc0, 0x1e, 0xce, 0xf0, 0x96, 0x7a,
		},
	},
	{
		.in = "abc",
		.out = {
			0x53, 0x04, 0x8e, 0x26, 0x81, 0x94, 0x1e, 0xf9,
			0x9b, 0x2e, 0x29, 0xb7, 0x6b, 0x4c, 0x7d, 0xab,
			0xe4, 0xc2, 0xd0, 0xc6, 0x34, 0xfc, 0x6d, 0x46,
			0xe0, 0xe2, 0xf1, 0x31, 0x07, 0xe7, 0xaf, 0x23,
		},
	},
	{
		.in = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
		    "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.out = {
			0x39, 0x28, 0xe1, 0x84, 0xfb, 0x86, 0x90, 0xf8,
			0x40, 0xda, 0x39, 0x88, 0x12, 0x1d, 0x31, 0xbe,
			0x65, 0xcb, 0x9d, 0x3e, 0xf8, 0x3e, 0xe6, 0x14,
			0x6f, 0xea, 0xc8, 0x61, 0xe1, 0x9b, 0x56, 0x3a,
		},
	},
	{
		.in = "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567"
		      "0123456701234567012345670123456701234567012345670123456701234567",
		.out = {
			0xcf, 0x78, 0xe4, 0xba, 0x93, 0x5b, 0x4d, 0x9e,
			0xb9, 0x10, 0x52, 0xae, 0xdd, 0xf8, 0xe2, 0xd6,
			0x06, 0xc5, 0x90, 0xf7, 0x08, 0x57, 0x36, 0x93,
			0xea, 0x94, 0xbe, 0x82, 0x6a, 0x66, 0x6e, 0xe4,
		},
	},
};

#define N_SHA512_256_TESTS (sizeof(sha512_256_tests) / sizeof(sha512_256_tests[0]))

static void
sha512_256_init(void *ctx)
{
	SHA512_256Init(ctx);
}

static void
sha512_256_update(void *ctx, const uint8_t *data, size_t len)
{
	SHA512_256Update(ctx, data, len);
}

static void
sha512_256_final(void *digest, void *ctx)
{
	SHA512_256Final(digest, ctx);
}

struct hash_ctx {
	uint8_t		*digest;
	size_t		 digest_len;
	void		*ctx;
	void		(*init)(void *);
	void		(*update)(void *, const uint8_t *, size_t);
	void		(*final)(void *, void *);
};

static const struct hash_tests {
	const char			*name;
	size_t				 num_tests;
	const struct hash_test_case	*tests;
} hash_tests[] = {
	[hash_md5] = {
		.name = "RFC 1321 MD5",
		.num_tests = N_MD5_TESTS,
		.tests = md5_tests,
	},
	[hash_rmd160] = {
		.name = "Bosselaers RMD160",
		.num_tests = N_RMD160_TESTS,
		.tests = rmd160_tests,
	},
	[hash_sha1] = {
		.name = "RFC 3174 SHA1",
		.num_tests = N_SHA1_TESTS,
		.tests = sha1_tests,
	},
	[hash_sha224] = {
		.name = "RFC 6234 SHA224",
		.num_tests = N_SHA224_TESTS,
		.tests = sha224_tests,
	},
	[hash_sha256] = {
		.name = "RFC 6234 SHA256",
		.num_tests = N_SHA256_TESTS,
		.tests = sha256_tests,
	},
	[hash_sha384] = {
		.name = "RFC 6234 SHA384",
		.num_tests = N_SHA384_TESTS,
		.tests = sha384_tests,
	},
	[hash_sha512] = {
		.name = "RFC 6234 SHA512",
		.num_tests = N_SHA512_TESTS,
		.tests = sha512_tests,
	},
	[hash_sha512_256] = {
		.name = "RFC 6234 SHA512_256 (generated)",
		.num_tests = N_SHA512_256_TESTS,
		.tests = sha512_256_tests,
	},
};

static int
hash_test_case(struct hash_ctx *ctx, const struct hash_test_case *tc,
    const char *name, size_t testno)
{
	size_t in_len = tc->in != NULL ? strlen(tc->in) : 0;

	ctx->init(ctx->ctx);
	ctx->update(ctx->ctx, (const uint8_t *)tc->in, in_len);
	ctx->final(ctx->digest, ctx->ctx);

	if (memcmp(tc->out, ctx->digest, ctx->digest_len) != 0) {
		fprintf(stderr, "FAIL: %s test %zu\n", name, testno);
		return 1;
	}

	return 0;
}

static int
hash_test(struct hash_ctx *ctx, const struct hash_tests *tests)
{
	size_t i;
	int failed = 0;

	for (i = 0; i < tests->num_tests; i++) {
		const struct hash_test_case *tc = &tests->tests[i];

		failed |= hash_test_case(ctx, tc, tests->name, i);
	}

	return failed;
}

int
main(void)
{
	uint8_t		md5_digest[MD5_DIGEST_LENGTH];
	uint8_t		rmd160_digest[RMD160_DIGEST_LENGTH];
	uint8_t		sha1_digest[SHA1_DIGEST_LENGTH];
	uint8_t		sha224_digest[SHA224_DIGEST_LENGTH];
	uint8_t		sha256_digest[SHA256_DIGEST_LENGTH];
	uint8_t		sha384_digest[SHA384_DIGEST_LENGTH];
	uint8_t		sha512_digest[SHA512_DIGEST_LENGTH];
	uint8_t		sha512_256_digest[SHA512_256_DIGEST_LENGTH];
	MD5_CTX		md5_ctx;
	RMD160_CTX	rmd160_ctx;
	SHA1_CTX	sha1_ctx;
	SHA2_CTX	sha224_ctx;
	SHA2_CTX	sha256_ctx;
	SHA2_CTX	sha384_ctx;
	SHA2_CTX	sha512_ctx;
	SHA2_CTX	sha512_256_ctx;
	struct hash_ctx	ctx[] = {
		[hash_md5] = {
			.digest = md5_digest,
			.digest_len = sizeof(md5_digest),
			.ctx = &md5_ctx,
			.init = md5_init,
			.update = md5_update,
			.final = md5_final,
		},
		[hash_rmd160] = {
			.digest = rmd160_digest,
			.digest_len = sizeof(rmd160_digest),
			.ctx = &rmd160_ctx,
			.init = rmd160_init,
			.update = rmd160_update,
			.final = rmd160_final,
		},
		[hash_sha1] = {
			.digest = sha1_digest,
			.digest_len = sizeof(sha1_digest),
			.ctx = &sha1_ctx,
			.init = sha1_init,
			.update = sha1_update,
			.final = sha1_final,
		},
		[hash_sha224] = {
			.digest = sha224_digest,
			.digest_len = sizeof(sha224_digest),
			.ctx = &sha224_ctx,
			.init = sha224_init,
			.update = sha224_update,
			.final = sha224_final,
		},
		[hash_sha256] = {
			.digest = sha256_digest,
			.digest_len = sizeof(sha256_digest),
			.ctx = &sha256_ctx,
			.init = sha256_init,
			.update = sha256_update,
			.final = sha256_final,
		},
		[hash_sha384] = {
			.digest = sha384_digest,
			.digest_len = sizeof(sha384_digest),
			.ctx = &sha384_ctx,
			.init = sha384_init,
			.update = sha384_update,
			.final = sha384_final,
		},
		[hash_sha512] = {
			.digest = sha512_digest,
			.digest_len = sizeof(sha512_digest),
			.ctx = &sha512_ctx,
			.init = sha512_init,
			.update = sha512_update,
			.final = sha512_final,
		},
		[hash_sha512_256] = {
			.digest = sha512_256_digest,
			.digest_len = sizeof(sha512_256_digest),
			.ctx = &sha512_256_ctx,
			.init = sha512_256_init,
			.update = sha512_256_update,
			.final = sha512_256_final,
		},
	};
	int i;
	int failed = 0;

	for (i = 0; i < NUM_HASHES; i++)
		failed |= hash_test(&ctx[i], &hash_tests[i]);

	return failed;
}
